import { Head, Link } from '@inertiajs/react';
import { Topbar } from '@/components/Topbar';
import { InfoBar } from '@/components/InfoBar';
import { Footer } from '@/components/Footer';
import { WhatsAppChat } from '@/components/WhatsAppChat';
import { CheckCircle, Calendar, Phone, Mail, ArrowLeft, Download } from 'lucide-react';

interface Machine {
    id: number;
    name: string;
    brand: string;
}

interface Quotation {
    id: number;
    visitor_name: string;
    visitor_email: string;
    visitor_phone: string;
    start_date: string;
    end_date: string;
    status: string;
    machines: Machine[];
}

interface Props {
    quotation: Quotation;
}

export default function QuotationsSuccess({ quotation }: Props) {
    const whatsappMessage = encodeURIComponent(
        `Hola, acabo de solicitar una cotización (ID: ${quotation.id}) para las siguientes máquinas:\n${quotation.machines.map((m) => `- ${m.name} (${m.brand})`).join('\n')}\n\nFechas: ${new Date(quotation.start_date).toLocaleDateString('es-ES')} - ${new Date(quotation.end_date).toLocaleDateString('es-ES')}`
    );
    const whatsappUrl = `https://wa.me/?text=${whatsappMessage}`;

    return (
        <>
            <Head title="Cotización Enviada - Akiyan" />
            <InfoBar />
            <Topbar />
            <div className="min-h-screen bg-gray-50">
                {/* Header */}
                <div className="relative overflow-hidden shadow-sm">
                    <div
                        className="absolute inset-0 bg-cover bg-center bg-no-repeat"
                        style={{
                            backgroundImage: 'url(/storage/fondo1.svg)',
                            filter: 'grayscale(100%)',
                        }}
                    >
                        <div className="absolute inset-0 bg-gradient-to-br from-[#262727]/40 via-[#2a2a2b]/30 to-[#30292A]/40" />
                    </div>
                    <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
                        <h1 className="text-2xl font-bold text-white mb-1">
                            ¡Cotización Enviada!
                        </h1>
                    </div>
                </div>

                <div className="max-w-3xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
                    <div className="bg-white rounded-xl shadow-lg p-8 text-center">
                        {/* Success Icon */}
                        <div className="flex justify-center mb-6">
                            <div className="w-20 h-20 bg-[#00E700]/10 rounded-full flex items-center justify-center">
                                <CheckCircle className="w-12 h-12 text-[#00E700]" />
                            </div>
                        </div>

                        <h2 className="text-2xl font-bold text-[#262727] mb-4">
                            Tu cotización ha sido enviada exitosamente
                        </h2>
                        <p className="text-[#30292A] mb-8">
                            Hemos recibido tu solicitud de cotización. Nuestro equipo se pondrá en contacto contigo pronto.
                        </p>

                        {/* Quotation Details */}
                        <div className="bg-gray-50 rounded-lg p-6 mb-8 text-left">
                            <h3 className="font-semibold text-[#262727] mb-4">
                                Detalles de tu Cotización
                            </h3>
                            <div className="space-y-3">
                                <div className="flex items-start gap-3">
                                    <Calendar className="w-5 h-5 text-[#00E700] mt-0.5" />
                                    <div>
                                        <p className="text-sm text-[#30292A]">
                                            <span className="font-medium">Rango de Fechas:</span>
                                        </p>
                                        <p className="text-[#262727]">
                                            {new Date(quotation.start_date).toLocaleDateString('es-ES')} - {new Date(quotation.end_date).toLocaleDateString('es-ES')}
                                        </p>
                                    </div>
                                </div>

                                <div className="flex items-start gap-3">
                                    <CheckCircle className="w-5 h-5 text-[#00E700] mt-0.5" />
                                    <div>
                                        <p className="text-sm text-[#30292A]">
                                            <span className="font-medium">Máquinas ({quotation.machines.length}):</span>
                                        </p>
                                        <ul className="list-disc list-inside text-[#262727] mt-1">
                                            {quotation.machines.map((machine) => (
                                                <li key={machine.id}>
                                                    {machine.name} ({machine.brand})
                                                </li>
                                            ))}
                                        </ul>
                                    </div>
                                </div>

                                <div className="flex items-start gap-3">
                                    <Mail className="w-5 h-5 text-[#00E700] mt-0.5" />
                                    <div>
                                        <p className="text-sm text-[#30292A]">
                                            <span className="font-medium">Email de contacto:</span>
                                        </p>
                                        <p className="text-[#262727]">{quotation.visitor_email}</p>
                                    </div>
                                </div>

                                <div className="flex items-start gap-3">
                                    <Phone className="w-5 h-5 text-[#00E700] mt-0.5" />
                                    <div>
                                        <p className="text-sm text-[#30292A]">
                                            <span className="font-medium">Teléfono:</span>
                                        </p>
                                        <p className="text-[#262727]">{quotation.visitor_phone}</p>
                                    </div>
                                </div>
                            </div>
                        </div>

                        {/* Actions */}
                        <div className="space-y-4">
                            <a
                                href={`/cotizar/pdf/${quotation.id}`}
                                target="_blank"
                                className="block w-full bg-[#00E700] hover:bg-[#00D600] text-white font-bold py-4 px-6 rounded-lg text-center transition-colors flex items-center justify-center gap-3"
                            >
                                <Download className="w-6 h-6" />
                                Descargar Comprobante PDF
                            </a>

                            <a
                                href={whatsappUrl}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="block w-full bg-green-600 hover:bg-green-700 text-white font-bold py-4 px-6 rounded-lg text-center transition-colors flex items-center justify-center gap-3"
                            >
                                <svg
                                    className="w-6 h-6"
                                    fill="currentColor"
                                    viewBox="0 0 24 24"
                                >
                                    <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893a11.821 11.821 0 00-3.48-8.413Z" />
                                </svg>
                                Contactar por WhatsApp
                            </a>

                            <Link
                                href="/maquinaria"
                                className="block w-full bg-gray-200 hover:bg-gray-300 text-[#262727] font-medium py-3 px-6 rounded-lg text-center transition-colors flex items-center justify-center gap-2"
                            >
                                <ArrowLeft className="w-5 h-5" />
                                Volver al Catálogo
                            </Link>
                        </div>
                    </div>
                </div>
            </div>
            <Footer />
            <WhatsAppChat />
        </>
    );
}

