import { Head, useForm, router } from '@inertiajs/react';
import { useState } from 'react';
import { Topbar } from '@/components/Topbar';
import { InfoBar } from '@/components/InfoBar';
import { Footer } from '@/components/Footer';
import { WhatsAppChat } from '@/components/WhatsAppChat';
import { Calendar, ChevronRight, ChevronLeft, Check, X } from 'lucide-react';

interface Machine {
    id: number;
    name: string;
    slug: string;
    category: string;
    brand: string;
    gallery_json: string[] | null;
}

interface CategoryWithCount {
    name: string;
    count: number;
}

interface Props {
    machines: Machine[];
    categoriesWithCount: CategoryWithCount[];
}

type Step = 1 | 2 | 3 | 4;

export default function QuotationsCreate({ machines, categoriesWithCount }: Props) {
    const [currentStep, setCurrentStep] = useState<Step>(1);
    const [selectedMachines, setSelectedMachines] = useState<number[]>([]);
    const [searchTerm, setSearchTerm] = useState('');
    const [selectedCategory, setSelectedCategory] = useState<string | null>(null);

    const { data, setData, post, processing, errors } = useForm({
        visitor_name: '',
        visitor_email: '',
        visitor_phone: '',
        start_date: '',
        end_date: '',
        machine_ids: [] as number[],
        notes: '',
    });

    const filteredMachines = machines.filter((machine) => {
        const matchesSearch = 
            machine.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
            machine.brand.toLowerCase().includes(searchTerm.toLowerCase()) ||
            machine.category.toLowerCase().includes(searchTerm.toLowerCase());
        
        const matchesCategory = !selectedCategory || machine.category === selectedCategory;
        
        return matchesSearch && matchesCategory;
    });

    const handleMachineToggle = (machineId: number) => {
        setSelectedMachines((prev) => {
            const newSelection = prev.includes(machineId)
                ? prev.filter((id) => id !== machineId)
                : [...prev, machineId];
            setData('machine_ids', newSelection);
            return newSelection;
        });
    };

    const handleNext = () => {
        if (currentStep === 1) {
            if (!data.start_date || !data.end_date) {
                return;
            }
            if (new Date(data.end_date) <= new Date(data.start_date)) {
                return;
            }
        }
        if (currentStep === 2) {
            if (selectedMachines.length === 0) {
                return;
            }
        }
        if (currentStep === 3) {
            if (!data.visitor_name || !data.visitor_email || !data.visitor_phone) {
                return;
            }
        }
        if (currentStep < 4) {
            setCurrentStep((prev) => (prev + 1) as Step);
        }
    };

    const handleBack = () => {
        if (currentStep > 1) {
            setCurrentStep((prev) => (prev - 1) as Step);
        }
    };

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        post('/cotizar');
    };

    const getDaysDifference = () => {
        if (data.start_date && data.end_date) {
            const start = new Date(data.start_date);
            const end = new Date(data.end_date);
            const diffTime = Math.abs(end.getTime() - start.getTime());
            const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
            return diffDays;
        }
        return 0;
    };

    const getSelectedMachinesData = () => {
        return machines.filter((m) => selectedMachines.includes(m.id));
    };

    const minDate = new Date().toISOString().split('T')[0];

    return (
        <>
            <Head title="Solicitar Cotización - Akiyan" />
            <InfoBar />
            <Topbar />
            <div className="min-h-screen bg-gray-50">
                {/* Header */}
                <div className="relative overflow-hidden shadow-sm">
                    <div
                        className="absolute inset-0 bg-cover bg-center bg-no-repeat"
                        style={{
                            backgroundImage: 'url(/storage/fondo1.svg)',
                            filter: 'grayscale(100%)',
                        }}
                    >
                        <div className="absolute inset-0 bg-gradient-to-br from-[#262727]/40 via-[#2a2a2b]/30 to-[#30292A]/40" />
                    </div>
                    <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
                        <h1 className="text-2xl font-bold text-white mb-1">
                            Solicitar Cotización
                        </h1>
                        <p className="text-sm text-gray-200">
                            Completa los pasos para generar tu cotización
                        </p>
                    </div>
                </div>

                <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
                    {/* Progress Steps */}
                    <div className="mb-8">
                        <div className="flex items-center justify-between">
                            {[1, 2, 3, 4].map((step) => (
                                <div key={step} className="flex items-center flex-1">
                                    <div className="flex flex-col items-center">
                                        <div
                                            className={`w-12 h-12 rounded-full flex items-center justify-center font-bold transition-all ${
                                                currentStep >= step
                                                    ? 'bg-[#00E700] text-white'
                                                    : 'bg-gray-300 text-gray-600'
                                            }`}
                                        >
                                            {currentStep > step ? (
                                                <Check className="w-6 h-6" />
                                            ) : (
                                                step
                                            )}
                                        </div>
                                        <span
                                            className={`mt-2 text-sm font-medium ${
                                                currentStep >= step
                                                    ? 'text-[#00E700]'
                                                    : 'text-gray-500'
                                            }`}
                                        >
                                            {step === 1 && 'Fechas'}
                                            {step === 2 && 'Máquinas'}
                                            {step === 3 && 'Contacto'}
                                            {step === 4 && 'Confirmar'}
                                        </span>
                                    </div>
                                    {step < 4 && (
                                        <div
                                            className={`flex-1 h-1 mx-2 transition-all ${
                                                currentStep > step
                                                    ? 'bg-[#00E700]'
                                                    : 'bg-gray-300'
                                            }`}
                                        />
                                    )}
                                </div>
                            ))}
                        </div>
                    </div>

                    {/* Step Content */}
                    <div className="bg-white rounded-xl shadow-lg p-8">
                        {/* Step 1: Date Range */}
                        {currentStep === 1 && (
                            <div className="space-y-6">
                                <div>
                                    <h2 className="text-lg font-bold text-[#262727] mb-2">
                                        Selecciona el Rango de Fechas
                                    </h2>
                                    <p className="text-[#30292A]">
                                        Indica desde cuándo y hasta cuándo necesitas la maquinaria
                                    </p>
                                </div>

                                <div className="grid md:grid-cols-2 gap-6">
                                    <div>
                                        <label
                                            htmlFor="start_date"
                                            className="block text-sm font-medium text-[#262727] mb-2"
                                        >
                                            Fecha de Inicio
                                        </label>
                                        <div className="relative">
                                            <Calendar className="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
                                            <input
                                                type="date"
                                                id="start_date"
                                                value={data.start_date}
                                                onChange={(e) =>
                                                    setData('start_date', e.target.value)
                                                }
                                                min={minDate}
                                                className="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#00E700] focus:border-[#00E700] outline-none"
                                            />
                                        </div>
                                        {errors.start_date && (
                                            <p className="mt-1 text-sm text-red-600">
                                                {errors.start_date}
                                            </p>
                                        )}
                                    </div>

                                    <div>
                                        <label
                                            htmlFor="end_date"
                                            className="block text-sm font-medium text-[#262727] mb-2"
                                        >
                                            Fecha de Fin
                                        </label>
                                        <div className="relative">
                                            <Calendar className="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
                                            <input
                                                type="date"
                                                id="end_date"
                                                value={data.end_date}
                                                onChange={(e) =>
                                                    setData('end_date', e.target.value)
                                                }
                                                min={
                                                    data.start_date || minDate
                                                }
                                                className="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#00E700] focus:border-[#00E700] outline-none"
                                            />
                                        </div>
                                        {errors.end_date && (
                                            <p className="mt-1 text-sm text-red-600">
                                                {errors.end_date}
                                            </p>
                                        )}
                                    </div>
                                </div>

                                {data.start_date && data.end_date && (
                                    <div className="bg-[#00E700]/10 border border-[#00E700]/20 rounded-lg p-4">
                                        <p className="text-[#262727] font-medium">
                                            Duración del arriendo: <span className="text-[#00E700]">{getDaysDifference()} días</span>
                                        </p>
                                    </div>
                                )}
                            </div>
                        )}

                        {/* Step 2: Machine Selection */}
                        {currentStep === 2 && (
                            <div className="space-y-6">
                                <div>
                                    <h2 className="text-lg font-bold text-[#262727] mb-2">
                                        Selecciona las Máquinas
                                    </h2>
                                    <p className="text-[#30292A]">
                                        Puedes seleccionar una o varias máquinas para cotizar
                                    </p>
                                </div>

                                <div className="grid lg:grid-cols-4 gap-6">
                                    {/* Sidebar de Categorías */}
                                    <div className="lg:col-span-1">
                                        <div className="bg-gray-50 rounded-lg p-4 sticky top-4">
                                            <h3 className="text-sm font-bold text-[#262727] mb-3">
                                                Categorías
                                            </h3>
                                            <div className="space-y-2">
                                                <button
                                                    onClick={() => setSelectedCategory(null)}
                                                    className={`w-full text-left p-2 rounded-lg text-sm transition-colors ${
                                                        !selectedCategory
                                                            ? 'bg-[#00E700] text-white font-semibold'
                                                            : 'bg-white text-[#262727] hover:bg-[#00E700]/10'
                                                    }`}
                                                >
                                                    Todas ({machines.length})
                                                </button>
                                                {categoriesWithCount.map((category) => (
                                                    <button
                                                        key={category.name}
                                                        onClick={() => setSelectedCategory(
                                                            selectedCategory === category.name ? null : category.name
                                                        )}
                                                        className={`w-full text-left p-2 rounded-lg text-sm transition-colors ${
                                                            selectedCategory === category.name
                                                                ? 'bg-[#00E700] text-white font-semibold'
                                                                : 'bg-white text-[#262727] hover:bg-[#00E700]/10'
                                                        }`}
                                                    >
                                                        {category.name} ({category.count})
                                                    </button>
                                                ))}
                                            </div>
                                        </div>
                                    </div>

                                    {/* Máquinas */}
                                    <div className="lg:col-span-3 space-y-4">
                                        {/* Search */}
                                        <div>
                                            <input
                                                type="text"
                                                placeholder="Buscar máquinas por nombre, marca o categoría..."
                                                value={searchTerm}
                                                onChange={(e) => setSearchTerm(e.target.value)}
                                                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#00E700] focus:border-[#00E700] outline-none"
                                            />
                                        </div>

                                        {/* Machines Grid */}
                                        <div className="max-h-96 overflow-y-auto">
                                            <div className="grid md:grid-cols-2 gap-4">
                                        {filteredMachines.map((machine) => {
                                            const isSelected = selectedMachines.includes(
                                                machine.id
                                            );
                                            return (
                                                <div
                                                    key={machine.id}
                                                    onClick={() =>
                                                        handleMachineToggle(machine.id)
                                                    }
                                                    className={`relative p-4 border-2 rounded-lg cursor-pointer transition-all ${
                                                        isSelected
                                                            ? 'border-[#00E700] bg-[#00E700]/5'
                                                            : 'border-gray-200 hover:border-[#00E700]/50'
                                                    }`}
                                                >
                                                    <div className="flex items-start gap-4">
                                                        <div className="flex-shrink-0 w-20 h-20 bg-gray-200 rounded-lg overflow-hidden">
                                                            {machine.gallery_json &&
                                                            machine.gallery_json.length > 0 ? (
                                                                <img
                                                                    src={machine.gallery_json[0]}
                                                                    alt={machine.name}
                                                                    className="w-full h-full object-cover"
                                                                />
                                                            ) : (
                                                                <div className="w-full h-full flex items-center justify-center text-2xl">
                                                                    🚜
                                                                </div>
                                                            )}
                                                        </div>
                                                        <div className="flex-1 min-w-0">
                                                            <h3 className="font-bold text-[#262727] mb-1">
                                                                {machine.name}
                                                            </h3>
                                                            <p className="text-sm text-[#30292A]">
                                                                {machine.brand} • {machine.category}
                                                            </p>
                                                        </div>
                                                        {isSelected && (
                                                            <div className="flex-shrink-0">
                                                                <div className="w-6 h-6 bg-[#00E700] rounded-full flex items-center justify-center">
                                                                    <Check className="w-4 h-4 text-white" />
                                                                </div>
                                                            </div>
                                                        )}
                                                    </div>
                                                </div>
                                            );
                                            })}
                                            </div>
                                        </div>

                                        {selectedMachines.length > 0 && (
                                            <div className="bg-[#00E700]/10 border border-[#00E700]/20 rounded-lg p-4">
                                                <p className="text-[#262727] font-medium">
                                                    {selectedMachines.length} máquina(s) seleccionada(s)
                                                </p>
                                            </div>
                                        )}
                                    </div>
                                </div>
                            </div>
                        )}

                        {/* Step 3: Contact Information */}
                        {currentStep === 3 && (
                            <div className="space-y-6">
                                <div>
                                    <h2 className="text-lg font-bold text-[#262727] mb-2">
                                        Información de Contacto
                                    </h2>
                                    <p className="text-[#30292A]">
                                        Completa tus datos para que podamos contactarte
                                    </p>
                                </div>

                                <div className="space-y-4">
                                    <div>
                                        <label
                                            htmlFor="visitor_name"
                                            className="block text-sm font-medium text-[#262727] mb-2"
                                        >
                                            Nombre Completo *
                                        </label>
                                        <input
                                            type="text"
                                            id="visitor_name"
                                            value={data.visitor_name}
                                            onChange={(e) =>
                                                setData('visitor_name', e.target.value)
                                            }
                                            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#00E700] focus:border-[#00E700] outline-none"
                                            required
                                        />
                                        {errors.visitor_name && (
                                            <p className="mt-1 text-sm text-red-600">
                                                {errors.visitor_name}
                                            </p>
                                        )}
                                    </div>

                                    <div>
                                        <label
                                            htmlFor="visitor_email"
                                            className="block text-sm font-medium text-[#262727] mb-2"
                                        >
                                            Email *
                                        </label>
                                        <input
                                            type="email"
                                            id="visitor_email"
                                            value={data.visitor_email}
                                            onChange={(e) =>
                                                setData('visitor_email', e.target.value)
                                            }
                                            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#00E700] focus:border-[#00E700] outline-none"
                                            required
                                        />
                                        {errors.visitor_email && (
                                            <p className="mt-1 text-sm text-red-600">
                                                {errors.visitor_email}
                                            </p>
                                        )}
                                    </div>

                                    <div>
                                        <label
                                            htmlFor="visitor_phone"
                                            className="block text-sm font-medium text-[#262727] mb-2"
                                        >
                                            Teléfono *
                                        </label>
                                        <input
                                            type="tel"
                                            id="visitor_phone"
                                            value={data.visitor_phone}
                                            onChange={(e) =>
                                                setData('visitor_phone', e.target.value)
                                            }
                                            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#00E700] focus:border-[#00E700] outline-none"
                                            required
                                        />
                                        {errors.visitor_phone && (
                                            <p className="mt-1 text-sm text-red-600">
                                                {errors.visitor_phone}
                                            </p>
                                        )}
                                    </div>

                                    <div>
                                        <label
                                            htmlFor="notes"
                                            className="block text-sm font-medium text-[#262727] mb-2"
                                        >
                                            Notas Adicionales (Opcional)
                                        </label>
                                        <textarea
                                            id="notes"
                                            value={data.notes}
                                            onChange={(e) => setData('notes', e.target.value)}
                                            rows={4}
                                            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#00E700] focus:border-[#00E700] outline-none resize-none"
                                            placeholder="Comparte cualquier información adicional que consideres relevante..."
                                        />
                                        {errors.notes && (
                                            <p className="mt-1 text-sm text-red-600">
                                                {errors.notes}
                                            </p>
                                        )}
                                    </div>
                                </div>
                            </div>
                        )}

                        {/* Step 4: Confirmation */}
                        {currentStep === 4 && (
                            <div className="space-y-6">
                                <div>
                                    <h2 className="text-lg font-bold text-[#262727] mb-2">
                                        Confirma tu Cotización
                                    </h2>
                                    <p className="text-[#30292A]">
                                        Revisa los detalles antes de enviar
                                    </p>
                                </div>

                                <div className="space-y-4">
                                    {/* Dates */}
                                    <div className="bg-gray-50 rounded-lg p-4">
                                        <h3 className="font-semibold text-[#262727] mb-2">
                                            Rango de Fechas
                                        </h3>
                                        <p className="text-[#30292A]">
                                            Desde: <span className="font-medium">{new Date(data.start_date).toLocaleDateString('es-ES')}</span>
                                        </p>
                                        <p className="text-[#30292A]">
                                            Hasta: <span className="font-medium">{new Date(data.end_date).toLocaleDateString('es-ES')}</span>
                                        </p>
                                        <p className="text-sm text-[#30292A] mt-1">
                                            Duración: {getDaysDifference()} días
                                        </p>
                                    </div>

                                    {/* Machines */}
                                    <div className="bg-gray-50 rounded-lg p-4">
                                        <h3 className="font-semibold text-[#262727] mb-2">
                                            Máquinas Seleccionadas ({selectedMachines.length})
                                        </h3>
                                        <ul className="space-y-2">
                                            {getSelectedMachinesData().map((machine) => (
                                                <li
                                                    key={machine.id}
                                                    className="text-[#30292A] flex items-center gap-2"
                                                >
                                                    <Check className="w-4 h-4 text-[#00E700]" />
                                                    {machine.name} ({machine.brand})
                                                </li>
                                            ))}
                                        </ul>
                                    </div>

                                    {/* Contact */}
                                    <div className="bg-gray-50 rounded-lg p-4">
                                        <h3 className="font-semibold text-[#262727] mb-2">
                                            Información de Contacto
                                        </h3>
                                        <p className="text-[#30292A]">
                                            <span className="font-medium">Nombre:</span> {data.visitor_name}
                                        </p>
                                        <p className="text-[#30292A]">
                                            <span className="font-medium">Email:</span> {data.visitor_email}
                                        </p>
                                        <p className="text-[#30292A]">
                                            <span className="font-medium">Teléfono:</span> {data.visitor_phone}
                                        </p>
                                        {data.notes && (
                                            <div className="mt-2">
                                                <p className="font-medium text-[#262727]">Notas:</p>
                                                <p className="text-[#30292A]">{data.notes}</p>
                                            </div>
                                        )}
                                    </div>
                                </div>
                            </div>
                        )}

                        {/* Navigation Buttons */}
                        <div className="flex justify-between mt-8 pt-6 border-t border-gray-200">
                            <button
                                type="button"
                                onClick={handleBack}
                                disabled={currentStep === 1}
                                className={`px-6 py-3 rounded-lg font-medium transition-all ${
                                    currentStep === 1
                                        ? 'bg-gray-100 text-gray-400 cursor-not-allowed'
                                        : 'bg-gray-200 text-[#262727] hover:bg-gray-300'
                                }`}
                            >
                                <div className="flex items-center gap-2">
                                    <ChevronLeft className="w-5 h-5" />
                                    Anterior
                                </div>
                            </button>

                            {currentStep < 4 ? (
                                <button
                                    type="button"
                                    onClick={handleNext}
                                    className="px-6 py-3 bg-[#00E700] text-white rounded-lg font-medium hover:bg-[#00D600] transition-all flex items-center gap-2"
                                >
                                    Siguiente
                                    <ChevronRight className="w-5 h-5" />
                                </button>
                            ) : (
                                <form onSubmit={handleSubmit}>
                                    <button
                                        type="submit"
                                        disabled={processing}
                                        className="px-6 py-3 bg-[#00E700] text-white rounded-lg font-medium hover:bg-[#00D600] transition-all disabled:opacity-50 disabled:cursor-not-allowed"
                                    >
                                        {processing ? 'Enviando...' : 'Enviar Cotización'}
                                    </button>
                                </form>
                            )}
                        </div>
                    </div>
                </div>
            </div>
            <Footer />
            <WhatsAppChat />
        </>
    );
}

