import { Head, Link } from '@inertiajs/react';
import { Topbar } from '@/components/Topbar';
import { InfoBar } from '@/components/InfoBar';
import { Footer } from '@/components/Footer';
import { WhatsAppChat } from '@/components/WhatsAppChat';

interface Builder {
    id: number;
    name: string;
}

interface Project {
    id: number;
    title: string;
    slug: string;
    description: string | null;
    year: number | null;
    location: string | null;
    gallery_json: string[] | null;
    builder: Builder;
}

interface Props {
    project: Project;
}

export default function ProjectsShow({ project }: Props) {
    const whatsappMessage = encodeURIComponent(
        `Hola, quiero información sobre el proyecto: ${project.title}`
    );
    const whatsappUrl = `https://wa.me/?text=${whatsappMessage}`;

    return (
        <>
            <Head title={`${project.title} - Akiyan`} />
            <InfoBar />
            <Topbar />
            <div className="min-h-screen bg-gray-50 ">
                {/* Header */}
                <div className="bg-white  shadow-sm">
                    <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
                        <Link
                            href="/proyectos"
                            className="inline-flex items-center text-gray-600 text-[#30292A] hover:text-gray-900 dark:hover:text-white mb-3 text-sm"
                        >
                            <svg
                                className="w-4 h-4 mr-2"
                                fill="none"
                                stroke="currentColor"
                                viewBox="0 0 24 24"
                            >
                                <path
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                    strokeWidth={2}
                                    d="M15 19l-7-7 7-7"
                                />
                            </svg>
                            Volver al portafolio
                        </Link>
                        <div className="flex items-center gap-4 mb-2">
                            {project.year && (
                                <span className="text-sm font-semibold text-[#00E700] px-3 py-1 bg-blue-50 dark:bg-blue-900/20 rounded-full">
                                    {project.year}
                                </span>
                            )}
                            {project.location && (
                                <span className="text-sm text-gray-500 text-[#30292A]">
                                    📍 {project.location}
                                </span>
                            )}
                        </div>
                        <h1 className="text-2xl font-bold text-gray-900 text-[#262727]">
                            {project.title}
                        </h1>
                    </div>
                </div>

                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
                    <div className="grid lg:grid-cols-3 gap-12">
                        {/* Main Content */}
                        <div className="lg:col-span-2 space-y-8">
                            {/* Gallery */}
                            {project.gallery_json && project.gallery_json.length > 0 ? (
                                <div className="space-y-4">
                                    <div className="aspect-[16/9] rounded-xl overflow-hidden bg-gray-200 dark:bg-gray-700">
                                        <img
                                            src={project.gallery_json[0]}
                                            alt={project.title}
                                            className="w-full h-full object-cover"
                                        />
                                    </div>
                                    {project.gallery_json.length > 1 && (
                                        <div className="grid grid-cols-4 gap-4">
                                            {project.gallery_json.slice(1, 5).map((image, index) => (
                                                <div
                                                    key={index}
                                                    className="aspect-square rounded-lg overflow-hidden bg-gray-200 dark:bg-gray-700"
                                                >
                                                    <img
                                                        src={image}
                                                        alt={`${project.title} ${index + 2}`}
                                                        className="w-full h-full object-cover"
                                                    />
                                                </div>
                                            ))}
                                        </div>
                                    )}
                                </div>
                            ) : (
                                <div className="aspect-[16/9] rounded-xl bg-gray-200 dark:bg-gray-700 flex items-center justify-center">
                                    <div className="text-9xl">🏗️</div>
                                </div>
                            )}

                            {/* Description */}
                            {project.description && (
                                <div>
                                    <h2 className="text-2xl font-bold text-gray-900 text-[#262727] mb-4">
                                        Descripción del Proyecto
                                    </h2>
                                    <div className="bg-white  rounded-lg shadow-sm p-8">
                                        <p className="text-gray-600 text-[#30292A] leading-relaxed text-lg whitespace-pre-line">
                                            {project.description}
                                        </p>
                                    </div>
                                </div>
                            )}

                            {/* Project Details */}
                            <div className="bg-white  rounded-lg shadow-sm p-8">
                                <h2 className="text-2xl font-bold text-gray-900 text-[#262727] mb-6">
                                    Detalles del Proyecto
                                </h2>
                                <dl className="space-y-4">
                                    {project.year && (
                                        <div className="flex justify-between py-2 border-b border-gray-200 dark:border-gray-700">
                                            <dt className="font-medium text-gray-700 text-[#262727]">
                                                Año de Ejecución
                                            </dt>
                                            <dd className="text-gray-900 text-[#262727]">
                                                {project.year}
                                            </dd>
                                        </div>
                                    )}
                                    {project.location && (
                                        <div className="flex justify-between py-2 border-b border-gray-200 dark:border-gray-700">
                                            <dt className="font-medium text-gray-700 text-[#262727]">
                                                Ubicación
                                            </dt>
                                            <dd className="text-gray-900 text-[#262727]">
                                                {project.location}
                                            </dd>
                                        </div>
                                    )}
                                    <div className="flex justify-between py-2">
                                        <dt className="font-medium text-gray-700 text-[#262727]">
                                            Constructora
                                        </dt>
                                        <dd className="text-gray-900 text-[#262727]">
                                            {project.builder.name}
                                        </dd>
                                    </div>
                                </dl>
                            </div>
                        </div>

                        {/* Sidebar */}
                        <div>
                            {/* WhatsApp CTA */}
                            <div className="bg-gradient-to-br from-green-600 to-green-700 rounded-lg shadow-lg p-6 text-white mb-6">
                                <h3 className="text-xl font-bold mb-3">
                                    ¿Interesado en este proyecto?
                                </h3>
                                <p className="mb-4 text-green-100">
                                    Contáctanos para más información sobre nuestros servicios
                                </p>
                                <a
                                    href={whatsappUrl}
                                    target="_blank"
                                    rel="noopener noreferrer"
                                    className="block w-full bg-white text-green-600 font-bold py-3 px-4 rounded-lg text-center hover:bg-green-50 transition-colors flex items-center justify-center gap-2"
                                >
                                    <svg
                                        className="w-5 h-5"
                                        fill="currentColor"
                                        viewBox="0 0 24 24"
                                    >
                                        <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893a11.821 11.821 0 00-3.48-8.413Z" />
                                    </svg>
                                    Contactar por WhatsApp
                                </a>
                            </div>

                            {/* Back to Projects */}
                            <Link
                                href="/proyectos"
                                className="block w-full bg-white  text-gray-700 text-[#262727] font-semibold py-3 px-4 rounded-lg text-center hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors border border-gray-200 dark:border-gray-700"
                            >
                                Ver más proyectos
                            </Link>
                        </div>
                    </div>
                </div>
            </div>
            <Footer />
            <WhatsAppChat />
        </>
    );
}

