import { Head, Link, router } from '@inertiajs/react';
import { Topbar } from '@/components/Topbar';
import { InfoBar } from '@/components/InfoBar';
import { Footer } from '@/components/Footer';
import { WhatsAppChat } from '@/components/WhatsAppChat';

interface Project {
    id: number;
    title: string;
    slug: string;
    description: string | null;
    year: number | null;
    location: string | null;
    gallery_json: string[] | null;
    builder: {
        id: number;
        name: string;
    };
}

interface Props {
    projects: {
        data: Project[];
        links: any;
        current_page: number;
        last_page: number;
    };
}

export default function ProjectsIndex({ projects }: Props) {
    return (
        <>
            <Head title="Portafolio de Proyectos - Akiyan" />
            <InfoBar />
            <Topbar />
            <div className="min-h-screen bg-gray-50">
                {/* Header */}
                <div className="relative overflow-hidden shadow-sm">
                    {/* Background Image */}
                    <div
                        className="absolute inset-0 bg-cover bg-center bg-no-repeat"
                        style={{
                            backgroundImage: 'url(/storage/fondo1.svg)',
                            filter: 'grayscale(100%)',
                        }}
                    >
                        {/* Overlay with Gray Gradient using palette colors - more subtle to show image */}
                        <div className="absolute inset-0 bg-gradient-to-br from-[#262727]/40 via-[#2a2a2b]/30 to-[#30292A]/40" />
                        <div className="absolute inset-0 bg-gradient-to-r from-[#30292A]/30 via-transparent to-[#262727]/30" />
                        <div className="absolute inset-0 bg-gradient-to-t from-[#1a1a1b]/40 via-transparent to-transparent" />
                        
                        {/* Shine Effect */}
                        <div className="absolute inset-0 overflow-hidden">
                            <div 
                                className="absolute inset-0 bg-gradient-to-r from-transparent via-white/15 to-transparent -skew-x-12"
                                style={{
                                    animation: 'shine 4s infinite',
                                }}
                            />
                        </div>
                    </div>
                    <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
                        <div className="flex items-center justify-between">
                            <div>
                                <h1 className="text-2xl font-bold text-white mb-1">
                                    Portafolio de Proyectos
                                </h1>
                                <p className="text-sm text-gray-200">
                                    Obras ejecutadas con excelencia y compromiso
                                </p>
                            </div>
                            <div className="flex gap-4">
                                <Link
                                    href="/constructora"
                                    className="text-white/80 hover:text-[#00E700] transition-colors"
                                >
                                    ← Constructora
                                </Link>
                                <Link
                                    href="/"
                                    className="text-white/80 hover:text-[#00E700] transition-colors"
                                >
                                    Inicio
                                </Link>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Projects Grid */}
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
                    {projects.data.length === 0 ? (
                        <div className="text-center py-24">
                            <div className="bg-white rounded-xl shadow-lg p-12 max-w-2xl mx-auto">
                                <div className="text-6xl mb-6">🏗️</div>
                                <h2 className="text-2xl font-bold text-[#262727] mb-4">
                                    No hay proyectos disponibles
                                </h2>
                                <p className="text-[#30292A] mb-6">
                                    Estamos trabajando en actualizar nuestro portafolio. 
                                    Pronto podrás ver aquí nuestros proyectos destacados.
                                </p>
                                <Link
                                    href="/constructora"
                                    className="inline-flex items-center text-[#00E700] font-semibold hover:text-[#00C803] transition-colors"
                                >
                                    Conoce más sobre nosotros
                                    <svg
                                        className="w-5 h-5 ml-2"
                                        fill="none"
                                        stroke="currentColor"
                                        viewBox="0 0 24 24"
                                    >
                                        <path
                                            strokeLinecap="round"
                                            strokeLinejoin="round"
                                            strokeWidth={2}
                                            d="M9 5l7 7-7 7"
                                        />
                                    </svg>
                                </Link>
                            </div>
                        </div>
                    ) : (
                        <>
                            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                                {projects.data.map((project) => (
                                    <Link
                                        key={project.id}
                                        href={`/proyectos/${project.slug}`}
                                        className="group bg-white rounded-xl shadow-lg hover:shadow-2xl transition-all duration-300 overflow-hidden transform hover:-translate-y-2"
                                    >
                                        <div className="aspect-[4/3] bg-gray-200 relative overflow-hidden">
                                            {project.gallery_json && project.gallery_json.length > 0 ? (
                                                <img
                                                    src={project.gallery_json[0]}
                                                    alt={project.title}
                                                    className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
                                                />
                                            ) : (
                                                <div className="w-full h-full flex items-center justify-center text-6xl bg-gradient-to-br from-[#262727] to-[#30292A]">
                                                    🏗️
                                                </div>
                                            )}
                                        </div>
                                        <div className="p-6">
                                            <div className="flex items-center justify-between mb-2">
                                                {project.year && (
                                                    <span className="text-sm font-semibold text-[#00E700]">
                                                        {project.year}
                                                    </span>
                                                )}
                                                {project.location && (
                                                    <span className="text-sm text-[#30292A]">
                                                        📍 {project.location}
                                                    </span>
                                                )}
                                            </div>
                                            <h3 className="text-xl font-bold text-[#262727] mb-3 group-hover:text-[#00E700] transition-colors">
                                                {project.title}
                                            </h3>
                                            {project.description && (
                                                <p className="text-[#30292A] line-clamp-2 mb-4">
                                                    {project.description}
                                                </p>
                                            )}
                                            <div className="flex items-center text-[#00E700] font-semibold">
                                                <span>Ver proyecto</span>
                                                <svg
                                                    className="w-5 h-5 ml-2 group-hover:translate-x-1 transition-transform"
                                                    fill="none"
                                                    stroke="currentColor"
                                                    viewBox="0 0 24 24"
                                                >
                                                    <path
                                                        strokeLinecap="round"
                                                        strokeLinejoin="round"
                                                        strokeWidth={2}
                                                        d="M9 5l7 7-7 7"
                                                    />
                                                </svg>
                                            </div>
                                        </div>
                                    </Link>
                                ))}
                            </div>

                            {/* Pagination */}
                            {projects.last_page > 1 && (
                                <div className="mt-12 flex justify-center gap-2">
                                    {projects.links.map((link: any, index: number) => (
                                        <button
                                            key={index}
                                            onClick={() => {
                                                if (link.url) {
                                                    router.get(link.url);
                                                }
                                            }}
                                            disabled={!link.url}
                                            className={`px-4 py-2 rounded-lg font-medium transition-colors ${
                                                link.active
                                                    ? 'bg-[#00E700] text-white'
                                                    : link.url
                                                      ? 'bg-white text-[#262727] hover:bg-gray-100 shadow-sm'
                                                      : 'bg-gray-100 text-gray-400 cursor-not-allowed'
                                            }`}
                                            dangerouslySetInnerHTML={{ __html: link.label }}
                                        />
                                    ))}
                                </div>
                            )}
                        </>
                    )}
                </div>
            </div>
            <Footer />
            <WhatsAppChat />
            <style>{`
                @keyframes shine {
                    0% {
                        transform: translateX(-100%) skewX(-12deg);
                    }
                    50% {
                        transform: translateX(200%) skewX(-12deg);
                    }
                    100% {
                        transform: translateX(200%) skewX(-12deg);
                    }
                }
            `}</style>
        </>
    );
}
