import { Head, Link } from '@inertiajs/react';
import { SEO } from '@/components/SEO';
import { Topbar } from '@/components/Topbar';
import { InfoBar } from '@/components/InfoBar';
import { Footer } from '@/components/Footer';
import { WhatsAppChat } from '@/components/WhatsAppChat';

interface Machine {
    id: number;
    name: string;
    slug: string;
    category: string;
    brand: string;
    description: string | null;
    specs_json: Record<string, any> | null;
    applications_json: string[] | null;
    gallery_json: string[] | null;
}

interface Props {
    machine: Machine;
}

export default function MachinesShow({ machine }: Props) {

    const whatsappMessage = encodeURIComponent(
        `Hola, quiero cotizar la maquinaria: ${machine.name}`
    );
    const whatsappUrl = `https://wa.me/?text=${whatsappMessage}`;

    const seoDescription = `${machine.name} disponible para arriendo en San Carlos, Ñuble, Chile. ${machine.description || 'Maquinaria pesada profesional para construcción.'} Servicio de arriendo con entrega en San Carlos, Ñuble.`;
    const seoKeywords = `arriendo ${machine.name.toLowerCase()} san carlos ñuble, arriendo ${machine.category.toLowerCase()} ñuble, ${machine.brand} ${machine.name.toLowerCase()} san carlos, arriendo maquinaria ${machine.category.toLowerCase()} ñuble chile`;

    return (
        <>
            <SEO
                title={`Arriendo de ${machine.name} en San Carlos, Ñuble - Akiyan`}
                description={seoDescription}
                keywords={seoKeywords}
                canonical={typeof window !== 'undefined' ? `${window.location.origin}/maquinaria/${machine.slug}` : ''}
                structuredData={{
                    '@context': 'https://schema.org',
                    '@type': 'Product',
                    name: machine.name,
                    description: seoDescription,
                    brand: {
                        '@type': 'Brand',
                        name: machine.brand,
                    },
                    category: machine.category,
                    offers: {
                        '@type': 'Offer',
                        availability: 'https://schema.org/InStock',
                        priceCurrency: 'CLP',
                        areaServed: {
                            '@type': 'City',
                            name: 'San Carlos, Ñuble, Chile',
                        },
                    },
                }}
            />
            <Head title={`${machine.name} - Akiyan`} />
            <InfoBar />
            <Topbar />
            <div className="min-h-screen bg-gray-50">
                {/* Header */}
                <div className="relative overflow-hidden shadow-sm">
                    {/* Background Image */}
                    <div
                        className="absolute inset-0 bg-cover bg-center bg-no-repeat"
                        style={{
                            backgroundImage: 'url(/storage/fondo1.svg)',
                            filter: 'grayscale(100%)',
                        }}
                    >
                        {/* Overlay with Gray Gradient using palette colors - more subtle to show image */}
                        <div className="absolute inset-0 bg-gradient-to-br from-[#262727]/40 via-[#2a2a2b]/30 to-[#30292A]/40" />
                        <div className="absolute inset-0 bg-gradient-to-r from-[#30292A]/30 via-transparent to-[#262727]/30" />
                        <div className="absolute inset-0 bg-gradient-to-t from-[#1a1a1b]/40 via-transparent to-transparent" />
                        
                        {/* Shine Effect */}
                        <div className="absolute inset-0 overflow-hidden">
                            <div 
                                className="absolute inset-0 bg-gradient-to-r from-transparent via-white/15 to-transparent -skew-x-12"
                                style={{
                                    animation: 'shine 4s infinite',
                                }}
                            />
                        </div>
                    </div>
                    <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
                        <div className="flex items-center justify-between">
                            <div>
                                <Link
                                    href="/maquinaria"
                                    className="inline-flex items-center text-white/80 hover:text-[#00E700] mb-3 transition-colors text-sm"
                                >
                                    <svg
                                        className="w-4 h-4 mr-2"
                                        fill="none"
                                        stroke="currentColor"
                                        viewBox="0 0 24 24"
                                    >
                                        <path
                                            strokeLinecap="round"
                                            strokeLinejoin="round"
                                            strokeWidth={2}
                                            d="M15 19l-7-7 7-7"
                                        />
                                    </svg>
                                    Volver al catálogo
                                </Link>
                                <h1 className="text-2xl font-bold text-white mb-1">
                                    {machine.name}
                                </h1>
                            </div>
                        </div>
                    </div>
                </div>

                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
                    <div className="grid lg:grid-cols-2 gap-12">
                        {/* Gallery */}
                        <div>
                            {machine.gallery_json && machine.gallery_json.length > 0 ? (
                                <div className="space-y-4">
                                    <div className="aspect-[4/3] rounded-xl overflow-hidden bg-gray-200 bg-gray-200">
                                        <img
                                            src={machine.gallery_json[0]}
                                            alt={machine.name}
                                            className="w-full h-full object-cover"
                                        />
                                    </div>
                                    {machine.gallery_json.length > 1 && (
                                        <div className="grid grid-cols-4 gap-4">
                                            {machine.gallery_json.slice(1, 5).map((image, index) => (
                                                <div
                                                    key={index}
                                                    className="aspect-square rounded-lg overflow-hidden bg-gray-200 bg-gray-200"
                                                >
                                                    <img
                                                        src={image}
                                                        alt={`${machine.name} ${index + 2}`}
                                                        className="w-full h-full object-cover"
                                                    />
                                                </div>
                                            ))}
                                        </div>
                                    )}
                                </div>
                            ) : (
                                <div className="aspect-[4/3] rounded-xl bg-gray-200 bg-gray-200 flex items-center justify-center">
                                    <div className="text-9xl">🚜</div>
                                </div>
                            )}
                        </div>

                        {/* Details */}
                        <div className="space-y-8">
                            {/* Description */}
                            {machine.description && (
                                <div>
                                    <h2 className="text-lg font-bold text-gray-900 text-[#262727] mb-3">
                                        Descripción
                                    </h2>
                                    <p className="text-gray-600 text-[#30292A] leading-relaxed">
                                        {machine.description}
                                    </p>
                                </div>
                            )}

                            {/* Specifications */}
                            {machine.specs_json && Object.keys(machine.specs_json).length > 0 && (
                                <div>
                                    <h2 className="text-lg font-bold text-gray-900 text-[#262727] mb-3">
                                        Especificaciones Técnicas
                                    </h2>
                                    <div className="bg-white  rounded-lg shadow-sm p-6">
                                        <dl className="space-y-3">
                                            {Object.entries(machine.specs_json).map(([key, value]) => (
                                                <div
                                                    key={key}
                                                    className="flex justify-between py-2 border-b border-gray-200 border-[#30292A] last:border-0"
                                                >
                                                    <dt className="font-medium text-gray-700 text-[#262727]">
                                                        {key}
                                                    </dt>
                                                    <dd className="text-gray-900 text-[#262727]">
                                                        {String(value)}
                                                    </dd>
                                                </div>
                                            ))}
                                        </dl>
                                    </div>
                                </div>
                            )}

                            {/* Applications */}
                            {machine.applications_json &&
                                machine.applications_json.length > 0 && (
                                    <div>
                                    <h2 className="text-lg font-bold text-gray-900 text-[#262727] mb-3">
                                        Aplicaciones
                                    </h2>
                                        <div className="bg-white  rounded-lg shadow-sm p-6">
                                            <ul className="space-y-2">
                                                {machine.applications_json.map((application, index) => (
                                                    <li
                                                        key={index}
                                                        className="flex items-start text-gray-600 text-[#30292A]"
                                                    >
                                                        <svg
                                                            className="w-5 h-5 text-[#00E700] mr-2 mt-0.5 flex-shrink-0"
                                                            fill="none"
                                                            stroke="currentColor"
                                                            viewBox="0 0 24 24"
                                                        >
                                                            <path
                                                                strokeLinecap="round"
                                                                strokeLinejoin="round"
                                                                strokeWidth={2}
                                                                d="M5 13l4 4L19 7"
                                                            />
                                                        </svg>
                                                        {application}
                                                    </li>
                                                ))}
                                            </ul>
                                        </div>
                                    </div>
                                )}

                            {/* CTA Buttons */}
                            <div className="pt-6 space-y-3">
                                <Link
                                    href="/cotizar"
                                    className="block w-full bg-[#00E700] hover:bg-[#00D600] text-white font-bold py-4 px-6 rounded-lg text-center transition-colors flex items-center justify-center gap-3"
                                >
                                    <svg
                                        className="w-6 h-6"
                                        fill="none"
                                        stroke="currentColor"
                                        viewBox="0 0 24 24"
                                    >
                                        <path
                                            strokeLinecap="round"
                                            strokeLinejoin="round"
                                            strokeWidth={2}
                                            d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"
                                        />
                                    </svg>
                                    Solicitar Cotización
                                </Link>
                                <a
                                    href={whatsappUrl}
                                    target="_blank"
                                    rel="noopener noreferrer"
                                    className="block w-full bg-green-600 hover:bg-green-700 text-white font-bold py-4 px-6 rounded-lg text-center transition-colors flex items-center justify-center gap-3"
                                >
                                    <svg
                                        className="w-6 h-6"
                                        fill="currentColor"
                                        viewBox="0 0 24 24"
                                    >
                                        <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893a11.821 11.821 0 00-3.48-8.413Z" />
                                    </svg>
                                    Cotizar por WhatsApp
                                </a>
                            </div>
                        </div>
                    </div>
                </div>

                {/* SEO Content - Hidden but readable by search engines */}
                <div className="sr-only">
                    <h2>Arriendo de {machine.name} en San Carlos, Ñuble, Chile</h2>
                    <p>
                        {machine.name} disponible para arriendo en San Carlos, región de Ñuble, Chile. 
                        {machine.description || 'Maquinaria pesada profesional para construcción.'} 
                        Marca: {machine.brand}. Categoría: {machine.category}. 
                        Servicio de arriendo con entrega en San Carlos, Ñuble y toda la región.
                    </p>
                    <p>
                        Akiyan ofrece arriendo de {machine.name} en San Carlos, Ñuble. 
                        {machine.brand} {machine.category} profesional para proyectos de construcción, 
                        obras civiles, movimiento de tierra y trabajos de infraestructura en la región de Ñuble, Chile.
                    </p>
                </div>
            </div>
            <Footer />
            <WhatsAppChat />
            <style>{`
                @keyframes shine {
                    0% {
                        transform: translateX(-100%) skewX(-12deg);
                    }
                    50% {
                        transform: translateX(200%) skewX(-12deg);
                    }
                    100% {
                        transform: translateX(200%) skewX(-12deg);
                    }
                }
            `}</style>
        </>
    );
}

