import { Head, Link, router } from '@inertiajs/react';
import { SEO } from '@/components/SEO';
import { useState } from 'react';
import { Topbar } from '@/components/Topbar';
import { InfoBar } from '@/components/InfoBar';
import { Footer } from '@/components/Footer';
import { WhatsAppChat } from '@/components/WhatsAppChat';

interface Machine {
    id: number;
    name: string;
    slug: string;
    category: string;
    brand: string;
    description: string | null;
    gallery_json: string[] | null;
}

interface CategoryWithCount {
    name: string;
    count: number;
}

interface Props {
    machines: {
        data: Machine[];
        links: any;
        current_page: number;
        last_page: number;
    };
    categoriesWithCount: CategoryWithCount[];
    categories: string[];
    brands: string[];
    filters: {
        category?: string;
        brand?: string;
    };
}

export default function MachinesIndex({ machines, categoriesWithCount, filters }: Props) {
    const totalMachines = categoriesWithCount.reduce((sum, cat) => sum + cat.count, 0);
    const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);

    return (
        <>
            <SEO
                title="Arriendo de Maquinaria Pesada en San Carlos, Ñuble - Catálogo Completo"
                description="Catálogo completo de maquinaria pesada para arriendo en San Carlos, Ñuble, Chile. Miniexcavadoras, retroexcavadoras, cargadores frontales, compactadores, vibropisones y más. Más de 42 máquinas disponibles. Servicio profesional con más de 15 años de experiencia en Ñuble."
                keywords="arriendo maquinaria ñuble, arriendo maquinaria san carlos, maquinaria pesada ñuble, arriendo excavadoras san carlos, arriendo maquinaria construcción ñuble, maquinaria pesada san carlos chile, arriendo miniexcavadoras ñuble, arriendo retroexcavadoras san carlos, arriendo compactadores ñuble"
                canonical={`${window.location.origin}/maquinaria`}
            />
            <Head title="Catálogo de Maquinaria - Akiyan" />
            <InfoBar />
            <Topbar />
            <div className="min-h-screen bg-gray-50">
                {/* Header */}
                <div className="relative overflow-hidden shadow-sm">
                    {/* Background Image */}
                    <div
                        className="absolute inset-0 bg-cover bg-center bg-no-repeat"
                        style={{
                            backgroundImage: 'url(/storage/fondo1.svg)',
                            filter: 'grayscale(100%)',
                        }}
                    >
                        {/* Overlay with Gray Gradient using palette colors - more subtle to show image */}
                        <div className="absolute inset-0 bg-gradient-to-br from-[#262727]/40 via-[#2a2a2b]/30 to-[#30292A]/40" />
                        <div className="absolute inset-0 bg-gradient-to-r from-[#30292A]/30 via-transparent to-[#262727]/30" />
                        <div className="absolute inset-0 bg-gradient-to-t from-[#1a1a1b]/40 via-transparent to-transparent" />
                        
                        {/* Shine Effect */}
                        <div className="absolute inset-0 overflow-hidden">
                            <div 
                                className="absolute inset-0 bg-gradient-to-r from-transparent via-white/15 to-transparent -skew-x-12"
                                style={{
                                    animation: 'shine 4s infinite',
                                }}
                            />
                        </div>
                    </div>
                    <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4 sm:py-6">
                        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
                            <div>
                                <h1 className="text-xl sm:text-2xl font-bold text-white mb-1">
                                    Catálogo de Maquinaria
                                </h1>
                                <p className="text-xs sm:text-sm text-gray-200">
                                    Encuentra la maquinaria que necesitas
                                </p>
                            </div>
                            <div className="flex items-center gap-2 sm:gap-4">
                                <Link
                                    href="/cotizar"
                                    className="bg-[#00E700] hover:bg-[#00D600] text-white font-bold px-4 py-2 sm:px-6 sm:py-3 rounded-lg transition-colors flex items-center gap-2 text-sm sm:text-base"
                                >
                                    <svg
                                        className="w-4 h-4 sm:w-5 sm:h-5"
                                        fill="none"
                                        stroke="currentColor"
                                        viewBox="0 0 24 24"
                                    >
                                        <path
                                            strokeLinecap="round"
                                            strokeLinejoin="round"
                                            strokeWidth={2}
                                            d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"
                                        />
                                    </svg>
                                    <span className="hidden sm:inline">Solicitar Cotización</span>
                                    <span className="sm:hidden">Cotizar</span>
                                </Link>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Main Content with Sidebar */}
                <div className="w-full py-4 sm:py-8 lg:py-12 relative">
                    {/* Mobile Menu Overlay */}
                    {isMobileMenuOpen && (
                        <div
                            className="fixed inset-0 bg-black/50 z-40 lg:hidden"
                            onClick={() => setIsMobileMenuOpen(false)}
                        />
                    )}
                    
                    <div className="flex flex-col lg:flex-row gap-4 lg:gap-0">
                        {/* Mobile Filter Button */}
                        <div className="lg:hidden px-4 sm:px-6">
                            <button
                                onClick={() => setIsMobileMenuOpen(!isMobileMenuOpen)}
                                className="w-full bg-white rounded-lg shadow-md p-4 flex items-center justify-between border border-gray-200"
                            >
                                <div className="flex items-center gap-2">
                                    <svg className="w-5 h-5 text-[#00E700]" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 4a1 1 0 011-1h16a1 1 0 011 1v2.586a1 1 0 01-.293.707l-6.414 6.414a1 1 0 00-.293.707V17l-4 4v-6.586a1 1 0 00-.293-.707L3.293 7.293A1 1 0 013 6.586V4z" />
                                    </svg>
                                    <span className="font-semibold text-[#262727]">
                                        {filters.category ? `Categoría: ${filters.category}` : 'Todas las Categorías'}
                                    </span>
                                </div>
                                <svg 
                                    className={`w-5 h-5 text-[#262727] transition-transform ${isMobileMenuOpen ? 'rotate-180' : ''}`}
                                    fill="none" 
                                    stroke="currentColor" 
                                    viewBox="0 0 24 24"
                                >
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                                </svg>
                            </button>
                        </div>

                        {/* Sidebar - Categorías */}
                        <div className={`w-full lg:w-80 flex-shrink-0 ${isMobileMenuOpen ? 'fixed top-0 left-0 right-0 bottom-0 z-50 lg:relative lg:z-auto' : 'hidden lg:block'}`}>
                            <div className={`bg-white rounded-lg lg:rounded-r-xl shadow-lg p-4 sm:p-6 lg:sticky lg:top-24 lg:h-[calc(100vh-6rem)] lg:overflow-y-auto ${isMobileMenuOpen ? 'h-full overflow-y-auto mx-0' : 'mx-4 sm:mx-6 lg:mx-0'}`}>
                                <div className="flex items-center justify-between mb-4 lg:mb-4">
                                    <h2 className="text-base sm:text-lg font-bold text-[#262727]">
                                        Categorías
                                    </h2>
                                    <button
                                        onClick={() => setIsMobileMenuOpen(false)}
                                        className="lg:hidden text-[#262727] hover:text-[#00E700]"
                                    >
                                        <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                                        </svg>
                                    </button>
                                </div>
                                <div className="space-y-2 max-h-[60vh] lg:max-h-none overflow-y-auto">
                                    <button
                                        onClick={() => {
                                            router.get('/maquinaria', {}, {
                                                preserveScroll: true,
                                                preserveState: false,
                                            });
                                            setIsMobileMenuOpen(false);
                                        }}
                                        className={`w-full text-left p-3 rounded-lg transition-colors border ${
                                            !filters.category
                                                ? 'bg-[#00E700]/10 border-[#00E700]/20 text-[#00E700] font-semibold'
                                                : 'border-transparent hover:bg-[#00E700]/5 hover:border-[#00E700]/10 text-[#262727]'
                                        }`}
                                    >
                                        <div className="flex items-center justify-between">
                                            <span>Todas las Categorías</span>
                                            <span className="text-xs bg-gray-200 px-2 py-1 rounded-full">
                                                {totalMachines}
                                            </span>
                                        </div>
                                    </button>
                                    {categoriesWithCount.map((category) => (
                                        <button
                                            key={category.name}
                                            onClick={() => {
                                                const url = new URL(window.location.href);
                                                // Eliminar el parámetro page al cambiar de categoría
                                                url.searchParams.delete('page');
                                                if (filters.category === category.name) {
                                                    url.searchParams.delete('category');
                                                } else {
                                                    url.searchParams.set('category', category.name);
                                                }
                                                router.get(url.pathname + url.search, {}, {
                                                    preserveScroll: true,
                                                    preserveState: false,
                                                });
                                                setIsMobileMenuOpen(false);
                                            }}
                                            className={`w-full text-left p-3 rounded-lg transition-colors border ${
                                                filters.category === category.name
                                                    ? 'bg-[#00E700]/10 border-[#00E700]/20 text-[#00E700] font-semibold'
                                                    : 'border-transparent hover:bg-[#00E700]/5 hover:border-[#00E700]/10 text-[#262727]'
                                            }`}
                                        >
                                            <div className="flex items-center justify-between">
                                                <span>{category.name}</span>
                                                <span className="text-xs bg-gray-200 px-2 py-1 rounded-full">
                                                    {category.count}
                                                </span>
                                            </div>
                                        </button>
                                    ))}
                                </div>
                            </div>
                        </div>

                        {/* Machines Grid */}
                        <div className="flex-1 px-4 sm:px-6 lg:px-8">
                            {machines.data.length === 0 ? (
                                <div className="text-center py-12 sm:py-24">
                                    <p className="text-sm sm:text-base text-[#30292A]">
                                        No se encontraron máquinas.
                                    </p>
                                </div>
                            ) : (
                                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4 sm:gap-6 lg:gap-8">
                                    {machines.data.map((machine) => (
                                        <Link
                                            key={machine.id}
                                            href={`/maquinaria/${machine.slug}`}
                                            className="group bg-white rounded-lg sm:rounded-xl shadow-md sm:shadow-lg hover:shadow-xl sm:hover:shadow-2xl transition-all duration-300 overflow-hidden transform hover:-translate-y-1 sm:hover:-translate-y-2"
                                        >
                                            <div className="aspect-[4/3] bg-gray-200 relative overflow-hidden">
                                                {machine.gallery_json && machine.gallery_json.length > 0 ? (
                                                    <img
                                                        src={machine.gallery_json[0]}
                                                        alt={machine.name}
                                                        className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
                                                    />
                                                ) : (
                                                    <div className="w-full h-full flex items-center justify-center text-4xl sm:text-6xl">
                                                        🚜
                                                    </div>
                                                )}
                                            </div>
                                            <div className="p-4 sm:p-6">
                                                <div className="mb-2">
                                                    <span className="text-xs sm:text-sm font-semibold text-[#00E700]">
                                                        {machine.category}
                                                    </span>
                                                </div>
                                                <h3 className="text-sm sm:text-base font-bold text-[#262727] mb-2 group-hover:text-[#00E700] transition-colors line-clamp-2">
                                                    {machine.name}
                                                </h3>
                                                {machine.description && (
                                                    <p className="text-xs sm:text-sm text-[#30292A] line-clamp-2 mb-3 sm:mb-4">
                                                        {machine.description}
                                                    </p>
                                                )}
                                                <div className="flex items-center text-[#00E700] font-semibold text-xs sm:text-sm">
                                                    <span>Ver detalles</span>
                                                    <svg
                                                        className="w-4 h-4 sm:w-5 sm:h-5 ml-2 group-hover:translate-x-1 transition-transform"
                                                        fill="none"
                                                        stroke="currentColor"
                                                        viewBox="0 0 24 24"
                                                    >
                                                        <path
                                                            strokeLinecap="round"
                                                            strokeLinejoin="round"
                                                            strokeWidth={2}
                                                            d="M9 5l7 7-7 7"
                                                        />
                                                    </svg>
                                                </div>
                                            </div>
                                        </Link>
                                    ))}
                                </div>
                            )}

                            {/* Pagination */}
                            {machines.last_page > 1 && (
                                <div className="mt-6 sm:mt-12 flex justify-center items-center gap-1 sm:gap-2 flex-wrap">
                                    {machines.links.map((link: any, index: number) => {
                                        // Traducir etiquetas de paginación
                                        let label = link.label;
                                        if (label === '&laquo; Previous') {
                                            label = '« Anterior';
                                        } else if (label === 'Next &raquo;') {
                                            label = 'Siguiente »';
                                        } else if (label === '&laquo;') {
                                            label = '«';
                                        } else if (label === '&raquo;') {
                                            label = '»';
                                        }

                                        return (
                                            <button
                                                key={index}
                                                onClick={() => {
                                                    if (link.url) {
                                                        router.get(link.url, {}, {
                                                            preserveScroll: true,
                                                        });
                                                    }
                                                }}
                                                disabled={!link.url}
                                                className={`px-2 sm:px-4 py-2 rounded-lg text-xs sm:text-sm font-medium transition-colors ${
                                                    link.active
                                                        ? 'bg-[#00E700] text-white'
                                                        : link.url
                                                          ? 'bg-white text-[#262727] hover:bg-gray-100 shadow-sm'
                                                          : 'bg-gray-100 text-gray-400 cursor-not-allowed'
                                                }`}
                                                dangerouslySetInnerHTML={{ __html: label }}
                                            />
                                        );
                                    })}
                                </div>
                            )}
                        </div>
                    </div>
                </div>

                {/* SEO Content - Hidden but readable by search engines */}
                <div className="sr-only">
                    <h2>Arriendo de Maquinaria Pesada en San Carlos, Ñuble, Chile</h2>
                    <p>
                        Akiyan ofrece servicios profesionales de arriendo de maquinaria pesada en San Carlos, región de Ñuble, Chile. 
                        Contamos con más de 15 años de experiencia proporcionando soluciones de construcción y maquinaria para proyectos 
                        en toda la región de Ñuble. Nuestro catálogo incluye más de 42 máquinas disponibles para arriendo, incluyendo 
                        miniexcavadoras, retroexcavadoras, cargadores frontales, compactadores, vibropisones, betoneras, y equipos 
                        especializados para construcción.
                    </p>
                    <h3>Servicios de Arriendo de Maquinaria en Ñuble</h3>
                    <p>
                        Especializados en arriendo de maquinaria pesada para construcción en San Carlos y toda la región de Ñuble. 
                        Ofrecemos maquinaria de última generación para proyectos de construcción, obras civiles, movimiento de tierra, 
                        compactación, y trabajos de infraestructura. Nuestro servicio incluye entrega y recogida en San Carlos, Ñuble, 
                        con opciones de arriendo diario, semanal y mensual.
                    </p>
                    <h3>Tipos de Maquinaria Disponible en San Carlos, Ñuble</h3>
                    <p>
                        Disponemos de maquinaria pesada para arriendo en San Carlos, Ñuble: excavadoras, retroexcavadoras, miniexcavadoras, 
                        cargadores frontales, compactadores de suelo, vibropisones, rodillos compactadores, betoneras, generadores, 
                        compresores, y equipos especializados para construcción. Todas nuestras máquinas están en excelente estado y 
                        son operadas por personal calificado.
                    </p>
                    <h3>Zona de Cobertura: San Carlos, Ñuble, Chile</h3>
                    <p>
                        Servimos a clientes en San Carlos y toda la región de Ñuble, incluyendo Chillán, Bulnes, Yungay, Quirihue, 
                        y otras localidades. Ofrecemos servicio de transporte de maquinaria pesada y asesoría técnica para proyectos 
                        de construcción en la región.
                    </p>
                </div>
            </div>
            <Footer />
            <WhatsAppChat />
            <style>{`
                @keyframes shine {
                    0% {
                        transform: translateX(-100%) skewX(-12deg);
                    }
                    50% {
                        transform: translateX(200%) skewX(-12deg);
                    }
                    100% {
                        transform: translateX(200%) skewX(-12deg);
                    }
                }
            `}</style>
        </>
    );
}

