import { Head, Link } from '@inertiajs/react';
import { Topbar } from '@/components/Topbar';
import { InfoBar } from '@/components/InfoBar';
import { Footer } from '@/components/Footer';
import { WhatsAppChat } from '@/components/WhatsAppChat';

interface Builder {
    id: number;
    name: string;
    description: string | null;
    history: string | null;
    certifications_json: string[] | null;
    logo: string | null;
}

interface Project {
    id: number;
    title: string;
    slug: string;
    description: string | null;
    year: number | null;
    location: string | null;
    gallery_json: string[] | null;
}

interface Props {
    builder: Builder;
    projects: Project[];
}

export default function BuilderIndex({ builder, projects }: Props) {
    const whatsappMessage = encodeURIComponent(
        'Hola, quiero información sobre sus servicios de construcción'
    );
    const whatsappUrl = `https://wa.me/?text=${whatsappMessage}`;

    return (
        <>
            <Head title={`${builder.name} - Akiyan`} />
            <InfoBar />
            <Topbar />
            <div className="min-h-screen bg-gray-50">
                {/* Header */}
                <div className="relative overflow-hidden shadow-sm">
                    {/* Background Image */}
                    <div
                        className="absolute inset-0 bg-cover bg-center bg-no-repeat"
                        style={{
                            backgroundImage: 'url(/storage/fondo1.svg)',
                            filter: 'grayscale(100%)',
                        }}
                    >
                        {/* Overlay with Gray Gradient using palette colors - more subtle to show image */}
                        <div className="absolute inset-0 bg-gradient-to-br from-[#262727]/40 via-[#2a2a2b]/30 to-[#30292A]/40" />
                        <div className="absolute inset-0 bg-gradient-to-r from-[#30292A]/30 via-transparent to-[#262727]/30" />
                        <div className="absolute inset-0 bg-gradient-to-t from-[#1a1a1b]/40 via-transparent to-transparent" />
                        
                        {/* Shine Effect */}
                        <div className="absolute inset-0 overflow-hidden">
                            <div 
                                className="absolute inset-0 bg-gradient-to-r from-transparent via-white/15 to-transparent -skew-x-12"
                                style={{
                                    animation: 'shine 4s infinite',
                                }}
                            />
                        </div>
                    </div>
                    <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
                        <div className="flex items-center justify-between">
                            <div>
                                <h1 className="text-2xl font-bold text-white mb-1">
                                    {builder.name}
                                </h1>
                                <p className="text-sm text-gray-200">
                                    Soluciones constructivas integrales con experiencia y compromiso
                                </p>
                            </div>
                            <Link
                                href="/"
                                className="text-white hover:text-[#00E700] transition-colors"
                            >
                                ← Volver al inicio
                            </Link>
                        </div>
                    </div>
                </div>

                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
                    <div className="grid lg:grid-cols-3 gap-12">
                        {/* Main Content */}
                        <div className="lg:col-span-2 space-y-8">
                            {/* Description */}
                            <div>
                                <h2 className="text-xl font-bold text-[#262727] mb-4">
                                    Sobre Nosotros
                                </h2>
                                <div className="bg-white rounded-xl shadow-lg p-8">
                                    <p className="text-[#30292A] leading-relaxed text-lg">
                                        {builder.description || 'Constructora Akiyan es una empresa líder en el sector de la construcción en Chile, con más de 15 años de experiencia desarrollando proyectos de infraestructura, edificación y obras civiles de alta calidad.'}
                                    </p>
                                </div>
                            </div>

                            {/* Services */}
                            <div>
                                <h2 className="text-xl font-bold text-[#262727] mb-4">
                                    Nuestros Servicios
                                </h2>
                                <div className="grid md:grid-cols-2 gap-6">
                                    <div className="bg-white rounded-xl shadow-lg p-6">
                                        <div className="text-4xl mb-4">🏗️</div>
                                        <h3 className="text-base font-bold text-[#262727] mb-2">
                                            Construcción de Obras Civiles
                                        </h3>
                                        <p className="text-[#30292A] leading-relaxed">
                                            Ejecutamos proyectos de infraestructura vial, puentes, túneles y obras de ingeniería civil con los más altos estándares de calidad y seguridad.
                                        </p>
                                    </div>
                                    <div className="bg-white rounded-xl shadow-lg p-6">
                                        <div className="text-4xl mb-4">🏢</div>
                                        <h3 className="text-base font-bold text-[#262727] mb-2">
                                            Edificación Comercial y Residencial
                                        </h3>
                                        <p className="text-[#30292A] leading-relaxed">
                                            Desarrollamos proyectos de edificios comerciales, residenciales y mixtos, desde la planificación hasta la entrega final.
                                        </p>
                                    </div>
                                    <div className="bg-white rounded-xl shadow-lg p-6">
                                        <div className="text-4xl mb-4">🔧</div>
                                        <h3 className="text-base font-bold text-[#262727] mb-2">
                                            Remodelaciones y Ampliaciones
                                        </h3>
                                        <p className="text-[#30292A] leading-relaxed">
                                            Especialistas en remodelaciones integrales, ampliaciones y mejoras de infraestructura existente, respetando siempre los plazos acordados.
                                        </p>
                                    </div>
                                    <div className="bg-white rounded-xl shadow-lg p-6">
                                        <div className="text-4xl mb-4">📐</div>
                                        <h3 className="text-base font-bold text-[#262727] mb-2">
                                            Asesoría y Consultoría
                                        </h3>
                                        <p className="text-[#30292A] leading-relaxed">
                                            Ofrecemos servicios de consultoría técnica, estudios de factibilidad, presupuestos y gestión de proyectos constructivos.
                                        </p>
                                    </div>
                                </div>
                            </div>

                            {/* History */}
                            {builder.history && (
                                <div>
                                    <h2 className="text-xl font-bold text-[#262727] mb-4">
                                        Nuestra Historia
                                    </h2>
                                    <div className="bg-white rounded-xl shadow-lg p-8">
                                        <p className="text-[#30292A] leading-relaxed whitespace-pre-line text-lg">
                                            {builder.history}
                                        </p>
                                    </div>
                                </div>
                            )}

                            {/* Values */}
                            <div>
                                <h2 className="text-xl font-bold text-[#262727] mb-4">
                                    Nuestros Valores
                                </h2>
                                <div className="bg-white rounded-xl shadow-lg p-8">
                                    <div className="grid md:grid-cols-2 gap-6">
                                        <div>
                                            <h3 className="text-base font-bold text-[#00E700] mb-2">
                                                Excelencia
                                            </h3>
                                            <p className="text-[#30292A] leading-relaxed">
                                                Buscamos la perfección en cada detalle, superando las expectativas de nuestros clientes con trabajos de la más alta calidad.
                                            </p>
                                        </div>
                                        <div>
                                            <h3 className="text-base font-bold text-[#00E700] mb-2">
                                                Compromiso
                                            </h3>
                                            <p className="text-[#30292A] leading-relaxed">
                                                Cumplimos con los plazos establecidos y nos comprometemos con el éxito de cada proyecto que emprendemos.
                                            </p>
                                        </div>
                                        <div>
                                            <h3 className="text-base font-bold text-[#00E700] mb-2">
                                                Innovación
                                            </h3>
                                            <p className="text-[#30292A] leading-relaxed">
                                                Adoptamos tecnologías modernas y metodologías eficientes para optimizar procesos y resultados.
                                            </p>
                                        </div>
                                        <div>
                                            <h3 className="text-base font-bold text-[#00E700] mb-2">
                                                Seguridad
                                            </h3>
                                            <p className="text-[#30292A] leading-relaxed">
                                                Priorizamos la seguridad de nuestros trabajadores y el cumplimiento de todas las normativas vigentes.
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {/* Projects Preview */}
                            {projects.length > 0 && (
                                <div>
                                    <div className="flex items-center justify-between mb-6">
                                        <h2 className="text-3xl font-bold text-[#262727]">
                                            Proyectos Destacados
                                        </h2>
                                        <Link
                                            href="/proyectos"
                                            className="text-[#00E700] hover:text-[#00C803] font-semibold transition-colors"
                                        >
                                            Ver todos →
                                        </Link>
                                    </div>
                                    <div className="grid md:grid-cols-2 gap-6">
                                        {projects.map((project) => (
                                            <Link
                                                key={project.id}
                                                href={`/proyectos/${project.slug}`}
                                                className="group bg-white rounded-xl shadow-lg hover:shadow-2xl transition-all duration-300 overflow-hidden transform hover:-translate-y-2"
                                            >
                                                <div className="aspect-[4/3] bg-gray-200 relative overflow-hidden">
                                                    {project.gallery_json &&
                                                    project.gallery_json.length > 0 ? (
                                                        <img
                                                            src={project.gallery_json[0]}
                                                            alt={project.title}
                                                            className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
                                                        />
                                                    ) : (
                                                        <div className="w-full h-full flex items-center justify-center text-6xl">
                                                            🏗️
                                                        </div>
                                                    )}
                                                </div>
                                                <div className="p-6">
                                                    <div className="flex items-center justify-between mb-2">
                                                        {project.year && (
                                                            <span className="text-sm text-[#30292A]">
                                                                {project.year}
                                                            </span>
                                                        )}
                                                        {project.location && (
                                                            <span className="text-sm text-[#30292A]">
                                                                📍 {project.location}
                                                            </span>
                                                        )}
                                                    </div>
                                                    <h3 className="text-xl font-bold text-[#262727] mb-2 group-hover:text-[#00E700] transition-colors">
                                                        {project.title}
                                                    </h3>
                                                    {project.description && (
                                                        <p className="text-[#30292A] line-clamp-2">
                                                            {project.description}
                                                        </p>
                                                    )}
                                                </div>
                                            </Link>
                                        ))}
                                    </div>
                                </div>
                            )}
                        </div>

                        {/* Sidebar */}
                        <div className="space-y-6">
                            {/* Certifications */}
                            {builder.certifications_json &&
                                builder.certifications_json.length > 0 && (
                                    <div className="bg-white rounded-xl shadow-lg p-6">
                                        <h3 className="text-xl font-bold text-[#262727] mb-4">
                                            Certificaciones
                                        </h3>
                                        <ul className="space-y-3">
                                            {builder.certifications_json.map((cert, index) => (
                                                <li
                                                    key={index}
                                                    className="flex items-start text-[#30292A]"
                                                >
                                                    <svg
                                                        className="w-5 h-5 text-[#00E700] mr-2 mt-0.5 flex-shrink-0"
                                                        fill="none"
                                                        stroke="currentColor"
                                                        viewBox="0 0 24 24"
                                                    >
                                                        <path
                                                            strokeLinecap="round"
                                                            strokeLinejoin="round"
                                                            strokeWidth={2}
                                                            d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"
                                                        />
                                                    </svg>
                                                    <span className="text-sm">{cert}</span>
                                                </li>
                                            ))}
                                        </ul>
                                    </div>
                                )}

                            {/* Statistics */}
                            <div className="bg-gradient-to-br from-[#262727] to-[#30292A] rounded-xl shadow-lg p-6 text-white">
                                <h3 className="text-xl font-bold mb-4">Números que Hablan</h3>
                                <div className="space-y-4">
                                    <div>
                                        <div className="text-3xl font-bold text-[#00E700]">200+</div>
                                        <div className="text-sm text-gray-300">Proyectos Completados</div>
                                    </div>
                                    <div>
                                        <div className="text-3xl font-bold text-[#00E700]">15+</div>
                                        <div className="text-sm text-gray-300">Años de Experiencia</div>
                                    </div>
                                    <div>
                                        <div className="text-3xl font-bold text-[#00E700]">50+</div>
                                        <div className="text-sm text-gray-300">Profesionales en Equipo</div>
                                    </div>
                                </div>
                            </div>

                            {/* Contact CTA */}
                            <div className="bg-gradient-to-br from-[#00C803] to-[#00E700] rounded-xl shadow-lg p-6 text-white">
                                <h3 className="text-xl font-bold mb-3">¿Necesitas un Proyecto?</h3>
                                <p className="mb-4 text-white/90">
                                    Contáctanos y te ayudaremos a hacer realidad tu proyecto de construcción
                                </p>
                                <a
                                    href={whatsappUrl}
                                    target="_blank"
                                    rel="noopener noreferrer"
                                    className="block w-full bg-white text-[#00E700] font-bold py-3 px-4 rounded-lg text-center hover:bg-gray-100 transition-colors"
                                >
                                    Contactar por WhatsApp
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <Footer />
            <WhatsAppChat />
            <style>{`
                @keyframes shine {
                    0% {
                        transform: translateX(-100%) skewX(-12deg);
                    }
                    50% {
                        transform: translateX(200%) skewX(-12deg);
                    }
                    100% {
                        transform: translateX(200%) skewX(-12deg);
                    }
                }
            `}</style>
        </>
    );
}
