import { useState } from 'react';
import { X, Send } from 'lucide-react';

export function WhatsAppChat() {
    const [isOpen, setIsOpen] = useState(false);
    const [selectedMessage, setSelectedMessage] = useState('');
    const [customMessage, setCustomMessage] = useState('');

    const predefinedMessages = [
        'Hola, quiero información sobre arriendo de maquinaria',
        'Necesito cotizar maquinaria para mi proyecto',
        'Quiero conocer el portafolio de proyectos',
        'Necesito información sobre servicios de construcción',
        'Quiero solicitar una visita técnica',
    ];

    const handleSendMessage = (message: string) => {
        const phoneNumber = '56912345678'; // Reemplazar con el número real
        const encodedMessage = encodeURIComponent(message);
        const whatsappUrl = `https://wa.me/${phoneNumber}?text=${encodedMessage}`;
        window.open(whatsappUrl, '_blank');
        setIsOpen(false);
        setSelectedMessage('');
        setCustomMessage('');
    };

    const googleMapsUrl = 'https://www.google.com/maps/place/Construccion+y+Arriendo+de+Maquinarias+Akiyan+SPA/@-36.405255,-71.9644885,614m/data=!3m2!1e3!4b1!4m6!3m5!1s0x9668cdff1869c367:0x3da5007fcabdb62b!8m2!3d-36.405255!4d-71.9644885!16s%2Fg%2F11wj8twvqm?entry=ttu&g_ep=EgoyMDI1MTIwMi4wIKXMDSoASAFQAw%3D%3D';

    return (
        <>
            {/* Floating Google Maps Button */}
            <a
                href={googleMapsUrl}
                target="_blank"
                rel="noopener noreferrer"
                className="fixed bottom-24 right-6 z-50 bg-white hover:bg-gray-50 text-[#262727] p-4 rounded-full shadow-2xl transition-all duration-300 hover:scale-110 flex items-center justify-center group border-2 border-gray-200"
                aria-label="Ver ubicación en Google Maps"
            >
                <svg
                    className="w-8 h-8"
                    fill="currentColor"
                    viewBox="0 0 24 24"
                >
                    <path d="M12 2C8.13 2 5 5.13 5 9c0 5.25 7 13 7 13s7-7.75 7-13c0-3.87-3.13-7-7-7zm0 9.5c-1.38 0-2.5-1.12-2.5-2.5s1.12-2.5 2.5-2.5 2.5 1.12 2.5 2.5-1.12 2.5-2.5 2.5z"/>
                </svg>
            </a>

            {/* Floating WhatsApp Button */}
            <button
                onClick={() => setIsOpen(true)}
                className="fixed bottom-6 right-6 z-50 bg-[#00E700] hover:bg-[#00C803] text-white p-4 rounded-full shadow-2xl transition-all duration-300 hover:scale-110 flex items-center justify-center group"
                aria-label="Abrir chat de WhatsApp"
            >
                <svg
                    className="w-8 h-8"
                    fill="currentColor"
                    viewBox="0 0 24 24"
                >
                    <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893a11.821 11.821 0 00-3.48-8.413Z" />
                </svg>
                {!isOpen && (
                    <span className="absolute -top-1 -right-1 w-4 h-4 bg-red-500 rounded-full animate-pulse" />
                )}
            </button>

            {/* Chat Modal */}
            {isOpen && (
                <div className="fixed inset-0 z-50 flex items-end justify-end p-4 md:p-6 pointer-events-none">
                    <div className="w-full max-w-md bg-white rounded-2xl shadow-2xl flex flex-col pointer-events-auto animate-slide-up">
                        {/* Chat Header */}
                        <div className="bg-[#00E700] text-white p-4 rounded-t-2xl flex items-center justify-between">
                            <div className="flex items-center gap-3">
                                <div className="w-10 h-10 bg-white/20 rounded-full flex items-center justify-center">
                                    <svg
                                        className="w-6 h-6"
                                        fill="currentColor"
                                        viewBox="0 0 24 24"
                                    >
                                        <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893a11.821 11.821 0 00-3.48-8.413Z" />
                                    </svg>
                                </div>
                                <div>
                                    <h3 className="font-semibold">Akiyan</h3>
                                    <p className="text-xs text-white/90">En línea</p>
                                </div>
                            </div>
                            <button
                                onClick={() => setIsOpen(false)}
                                className="p-2 hover:bg-white/20 rounded-full transition-colors"
                                aria-label="Cerrar chat"
                            >
                                <X className="w-5 h-5" />
                            </button>
                        </div>

                        {/* Chat Body */}
                        <div className="flex-1 overflow-y-auto p-4 space-y-3 bg-gray-50 max-h-[400px]">
                            {/* Welcome Message */}
                            <div className="flex items-start gap-2">
                                <div className="w-8 h-8 bg-[#00E700] rounded-full flex items-center justify-center flex-shrink-0">
                                    <svg
                                        className="w-5 h-5 text-white"
                                        fill="currentColor"
                                        viewBox="0 0 24 24"
                                    >
                                        <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893a11.821 11.821 0 00-3.48-8.413Z" />
                                    </svg>
                                </div>
                                <div className="bg-white rounded-lg rounded-tl-none p-3 shadow-sm max-w-[85%]">
                                    <p className="text-sm text-[#262727]">
                                        ¡Hola! 👋 Bienvenido a Akiyan. ¿En qué podemos ayudarte hoy?
                                    </p>
                                </div>
                            </div>

                            {/* Predefined Messages */}
                            <div className="space-y-2">
                                <p className="text-xs text-gray-500 px-2">Mensajes rápidos:</p>
                                {predefinedMessages.map((message, index) => (
                                    <button
                                        key={index}
                                        onClick={() => {
                                            setSelectedMessage(message);
                                            setCustomMessage('');
                                        }}
                                        className={`w-full text-left p-3 rounded-lg border-2 transition-all ${
                                            selectedMessage === message
                                                ? 'border-[#00E700] bg-[#00E700]/5'
                                                : 'border-gray-200 bg-white hover:border-[#00E700]/50 hover:bg-gray-50'
                                        }`}
                                    >
                                        <p className="text-sm text-[#262727]">{message}</p>
                                    </button>
                                ))}
                            </div>

                            {/* Custom Message Input */}
                            <div className="pt-2">
                                <p className="text-xs text-gray-500 px-2 mb-2">O escribe tu mensaje:</p>
                                <textarea
                                    value={customMessage}
                                    onChange={(e) => {
                                        setCustomMessage(e.target.value);
                                        setSelectedMessage('');
                                    }}
                                    placeholder="Escribe tu mensaje aquí..."
                                    className="w-full p-3 border-2 border-gray-200 rounded-lg focus:border-[#00E700] focus:outline-none resize-none text-sm text-[#262727]"
                                    rows={3}
                                />
                            </div>
                        </div>

                        {/* Chat Footer */}
                        <div className="p-4 bg-white border-t border-gray-200 rounded-b-2xl">
                            <button
                                onClick={() => {
                                    const messageToSend = customMessage || selectedMessage;
                                    if (messageToSend) {
                                        handleSendMessage(messageToSend);
                                    }
                                }}
                                disabled={!selectedMessage && !customMessage}
                                className={`w-full flex items-center justify-center gap-2 py-3 px-4 rounded-lg font-semibold transition-all ${
                                    selectedMessage || customMessage
                                        ? 'bg-[#00E700] hover:bg-[#00C803] text-white'
                                        : 'bg-gray-200 text-gray-400 cursor-not-allowed'
                                }`}
                            >
                                <Send className="w-5 h-5" />
                                <span>Enviar por WhatsApp</span>
                            </button>
                        </div>
                    </div>
                </div>
            )}

            {/* Backdrop */}
            {isOpen && (
                <div
                    className="fixed inset-0 bg-black/20 z-40"
                    onClick={() => setIsOpen(false)}
                />
            )}

            <style>{`
                @keyframes slide-up {
                    from {
                        transform: translateY(100%);
                        opacity: 0;
                    }
                    to {
                        transform: translateY(0);
                        opacity: 1;
                    }
                }
                .animate-slide-up {
                    animation: slide-up 0.3s ease-out;
                }
            `}</style>
        </>
    );
}

