import { Head } from '@inertiajs/react';

interface SEOProps {
    title?: string;
    description?: string;
    keywords?: string;
    canonical?: string;
    ogImage?: string;
    structuredData?: object;
    noindex?: boolean;
}

export function SEO({
    title = 'Arriendo de Maquinaria en Ñuble, San Carlos - Akiyan',
    description = 'Arriendo de maquinaria pesada en San Carlos, Ñuble, Chile. Miniexcavadoras, retroexcavadoras, cargadores frontales, compactadores y más. Servicio profesional con más de 15 años de experiencia.',
    keywords = 'arriendo maquinaria ñuble, arriendo maquinaria san carlos, maquinaria pesada ñuble, arriendo excavadoras san carlos, arriendo maquinaria construcción ñuble, maquinaria pesada san carlos chile',
    canonical,
    ogImage = '/storage/logoAkiyan.png',
    structuredData,
    noindex = false,
}: SEOProps) {
    const fullTitle = title.includes('Akiyan') ? title : `${title} - Akiyan`;
    const fullDescription = description || 'Arriendo de maquinaria pesada en San Carlos, Ñuble, Chile. Servicio profesional con más de 15 años de experiencia.';
    
    // Structured Data para LocalBusiness
    const defaultStructuredData = {
        '@context': 'https://schema.org',
        '@type': 'LocalBusiness',
        name: 'Akiyan - Arriendo de Maquinaria',
        description: fullDescription,
        image: ogImage,
        address: {
            '@type': 'PostalAddress',
            streetAddress: 'San Carlos',
            addressLocality: 'San Carlos',
            addressRegion: 'Ñuble',
            addressCountry: 'CL',
        },
        geo: {
            '@type': 'GeoCoordinates',
            latitude: -36.405255,
            longitude: -71.9644885,
        },
        url: canonical || (typeof window !== 'undefined' ? window.location.href : ''),
        telephone: '+56912345678',
        priceRange: '$$',
        areaServed: {
            '@type': 'City',
            name: 'San Carlos, Ñuble, Chile',
        },
        serviceType: 'Arriendo de Maquinaria Pesada',
    };

    const finalStructuredData = structuredData || defaultStructuredData;

    return (
        <Head>
            <title>{fullTitle}</title>
            <meta name="description" content={fullDescription} />
            <meta name="keywords" content={keywords} />
            {noindex && <meta name="robots" content="noindex, nofollow" />}
            {!noindex && <meta name="robots" content="index, follow" />}
            <link rel="canonical" href={canonical || (typeof window !== 'undefined' ? window.location.href : '')} />
            
            {/* Open Graph / Facebook */}
            <meta property="og:type" content="website" />
            <meta property="og:url" content={canonical || (typeof window !== 'undefined' ? window.location.href : '')} />
            <meta property="og:title" content={fullTitle} />
            <meta property="og:description" content={fullDescription} />
            <meta property="og:image" content={ogImage} />
            <meta property="og:locale" content="es_CL" />
            
            {/* Twitter */}
            <meta name="twitter:card" content="summary_large_image" />
            <meta name="twitter:url" content={canonical || (typeof window !== 'undefined' ? window.location.href : '')} />
            <meta name="twitter:title" content={fullTitle} />
            <meta name="twitter:description" content={fullDescription} />
            <meta name="twitter:image" content={ogImage} />
            
            {/* Geo Tags */}
            <meta name="geo.region" content="CL-NB" />
            <meta name="geo.placename" content="San Carlos, Ñuble" />
            <meta name="geo.position" content="-36.405255;-71.9644885" />
            <meta name="ICBM" content="-36.405255, -71.9644885" />
            
            {/* Structured Data */}
            <script
                type="application/ld+json"
                dangerouslySetInnerHTML={{
                    __html: JSON.stringify(finalStructuredData),
                }}
            />
        </Head>
    );
}

